package jp.co.sample.util;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jp.co.sample.contants.StringConstants;
import jp.co.sample.exception.SystemErrorException;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * 文字列操作utilクラス。
 *
 * @author 生地 真規
 */
public class StringUtil {

	/** log */
	private static Log log = LogFactory.getLog(StringUtil.class);
	
	/** URLを抽出するための正規表現パターン */
	public static final String urlPattern = "(http://|https://){1}[\\w\\.\\-/:\\#\\?\\=\\&\\;\\%\\~\\+]+";

	
	
	/**
	 * html用のサニタイズ処理を行います。
	 * @param value サニタイズ処理対象文字列
	 * @return htmlサニタイズされた文字列
	 */
	public static String sanitize(String value) {
		if (log.isDebugEnabled()) log.debug("-------- start sanitize(String value)");

		if (StringUtils.isEmpty(value)) {
			if (log.isDebugEnabled()) log.debug("文字列が指定されていません。");
			return value;
		}
		if (log.isDebugEnabled()) log.debug("処理対象文字列[" + value + "]");

		StringBuilder result = new StringBuilder();
		String filtered = null;

		for (int i = 0; i < value.length(); i++) {
			filtered = null;

			switch (value.charAt(i)) {
			case '<':
				filtered = "&lt;";

				break;

			case '>':
				filtered = "&gt;";

				break;

			case '&':
				filtered = "&amp;";

				break;

			case '"':
				filtered = "&quot;";

				break;

			case '\'':
				filtered = "&#39;";

				break;
			}

			if (filtered == null) {
				result.append(value.charAt(i));
			} else {
				result.append(filtered);
			}
		}
		if (log.isDebugEnabled()) log.debug("処理後文字列[" + result.toString() + "]");
		return result.toString();
	}


	/**
	 * 文字列を指定文字数でカットし、変換後の文字列を返します。
	 * @param arg 文字列
	 * @param mojisu 文字数
	 * @return	指定された文字数で切られた文字列
	 */
	public static String substringByMojisu(String arg, int mojisu) {
		if (log.isDebugEnabled()) log.debug("-------- start substringByMojisu(String arg, int mojisu)");
		
		if (arg == null) {
			if (log.isDebugEnabled()) log.debug("文字列がnullです。");
			return null;
		}

		if (arg.length() > mojisu) {
			return arg.substring(0, mojisu);
		}
		return arg;
	}


	/**
	 * 数値に変換可能な文字列かどうか判定します。
	 * @param value 検査対象文字列
	 * @return  数値に変換可能ならtrue（空文字はfalse）
	 */
	public static boolean isNumeric(String value) {
		if (log.isDebugEnabled()) log.debug("-------- start isNumeric(String value)");
		
		if (StringUtils.isNotEmpty(value) && StringUtils.isNumeric(value)) {
			return true;
		}
		return false;
	}

	/**
	 * 指定された文字列が半角数字か判定します。
	 * @param value 検査対象文字列
	 * @return 半角数値のみであればtrue
	 */
	public static boolean isHankakuNumber(String value) {
		if (log.isDebugEnabled()) log.debug("-------- start isHankakuNumber(String value)");
		
		if (isNumeric(value)) {
			Pattern pattern = Pattern.compile("^[0-9]*$");
			Matcher matcher = pattern.matcher(value);
			boolean b = matcher.matches();
			return b;
		}
		return false;
	}
	
	/**
	 * 文字列を数値型（Integer）に変換します。変換できない場合、nullを返します。
	 * @param value 変換対象文字列
	 * @return Integer型に変換された数値
	 */
	public static Integer parseInteger(String value) {
		if (log.isDebugEnabled()) log.debug("-------- start parseInteger(String value)");
		
		if (isNumeric(value)) {
			return Integer.valueOf(value);
		}
		return null;
	}

	/**
	 * 文字列を数値型（Long）に変換します。変換できない場合、nullを返します。
	 * @param value 変換対象文字列
	 * @return Long型に変換された数値
	 */
	public static Long parseLong(String value) {
		if (log.isDebugEnabled()) log.debug("-------- start parseLong(String value)");
		
		if (isNumeric(value)) {
			return Long.valueOf(value);
		}
		return null;
	}

	/**
	 * 改行コードを<br/>タグに変換します。
	 * @param value 変換対象文字列
	 * @return html出力用に改行を<br/>タグに置換した文字列
	 */
	public static String nl2br(String value) {
		if (log.isDebugEnabled()) log.debug("-------- start nl2br(String value)");
		
		if (value == null) {
			if (log.isDebugEnabled()) log.debug("文字列が指定されていません。");
			return null;
		}
		String rtn = value.replaceAll("\\r\\n", "\n").replaceAll("\\r", "\n").replaceAll("\\n", "<br/>");
		return rtn;
	}
	
	/**
	 * 改行コード(CR, LF)を除去します。
	 * @param value 変換対象文字列
	 * @return 改行を除去した文字列
	 */
	public static String removeCrLf(String value) {
		if (log.isDebugEnabled()) log.debug("-------- start removeCrLf(String value)");
		
		if (value == null) {
			if (log.isDebugEnabled()) log.debug("文字列が指定されていません。");
			return null;
		}
		String rtn = value.replaceAll("\\r\\n", "").replaceAll("\\r", "").replaceAll("\\n", "");
		return rtn;
	}

	/**
	 * バイト数で文字列を切り出す。
	 * @param value 対象文字列
	 * @param length バイト長
	 * @param charset 文字セット名
	 * @return 切り出した文字列
	 */
	public static String substringByByte(String value, int length, String charset) {
		if (log.isDebugEnabled()) log.debug("-------- start substringByByte(String value, int length, String charset)");
		
		return substringByByte(value, length, charset, StringUtils.EMPTY);
	}

	/**
	 * バイト数で文字列を切り出す。
	 * @param value 対象文字列
	 * @param length バイト長
	 * @param charset 文字セット名
	 * @param overflowChar 桁あふれの時に表示する文字
	 * @return 切り出し後の文字列
	 */
	public static String substringByByte(String value, int length, String charset, String overflowChar) {
		if (log.isDebugEnabled()) log.debug("-------- start substringByByte(String value, int length, String charset, String overflowChar)");
		
		if (StringUtils.isEmpty(value)) {
			return value;
		}
		if (length == 0) {
			return StringUtils.EMPTY;
		}
		if (log.isDebugEnabled()) {
			log.debug("処理対象文字列[" + value + "]");
			log.debug("文字セット[" + charset + "]");
			log.debug("処理対象文字列のバイト長 = " + getByteLength(value, charset));
		}
		
        if (getByteLength(value, charset) > length) {
        	// 指定バイト数より長い場合

    		// 桁あふれの時に表示する文字のバイト数
    		int overflowLength = 0;
    		if (StringUtils.isNotEmpty(overflowChar)) {
    			overflowLength = getByteLength(overflowChar, charset);
    		} else {
    			overflowChar = StringUtils.EMPTY;
    		}
        	
        	StringBuilder sb = new StringBuilder();
	        char[] charArray = value.toCharArray();
	        for (int i = 0; i < charArray.length; i++) {
	        	
	        	int charlen = getByteLength(charArray[i], charset);

	        	if (getByteLength(sb, charset) + charlen + overflowLength > length) {
	        		sb.append(overflowChar);
	        		break;
	        	}
	        	sb.append(charArray[i]);
	        }
	        return sb.toString();
        }
        return value;
	}

	/**
	 * 指定した文字セットでの文字のバイト数を取得します。
	 * @param value 対象文字
	 * @param charset 文字セット名
	 * @return 指定した文字セットでの文字のバイト数
	 */
	public static int getByteLength(char value, String charset) {
		if (log.isDebugEnabled()) log.debug("-------- start getByteLength(char value, String charset)");
		
		return getByteLength(String.valueOf(value), charset);
	}

	/**
	 * 指定した文字セットでの文字列のバイト数を取得します。
	 * @param value 対象文字列
	 * @param charset 文字セット名
	 * @return 指定した文字セットでの文字列のバイト数
	 */
	public static int getByteLength(String value, String charset) {
		if (log.isDebugEnabled()) log.debug("-------- start getByteLength(String value, String charset)");

		if (value == null) {
			return 0;
		}
		if (StringUtils.isEmpty(charset)) {
			throw new SystemErrorException("文字セットが指定されていません。");
		}
		
		try {
			return value.getBytes(charset).length;
		} catch (UnsupportedEncodingException e) {
			throw new SystemErrorException(e);
		}
	}
	
	/**
	 * 指定した文字セットでの文字列のバイト数を取得します。
	 * @param value 対象文字列
	 * @param charset 文字セット名
	 * @return 指定した文字セットでの文字列のバイト数
	 */
	public static int getByteLength(StringBuilder value, String charset) {
		if (log.isDebugEnabled()) log.debug("-------- start getByteLength(StringBuilder value, String charset)");
		
		if (value == null) {
			return 0;
		}
		
		return getByteLength(String.valueOf(value), charset);
	}

	/**
	 * CSV(カンマ区切り値)1行分のデータをカンマで分割し、要素をリストに格納して返す。
	 * 無効な値の場合は、空のリストが返る。
	 * @param csv CSV(カンマ区切り値)
	 * @return Stringのリスト
	 */
	public static List<String> csvLineToList(String csv) {
		if (log.isDebugEnabled()) log.debug("-------- start csvLineToList(String csv)");
		
		return csvLineToList(csv, StringConstants.COMMA);
	}
	
	/**
	 * CSV(カンマ区切り値)1行分のデータをカンマで分割し、要素をリストに格納して返す。
	 * 無効な値の場合は、空のリストが返る。
	 * @param csv CSV
	 * @param sepalatorRegex 区切り文字 
	 * @return Stringのリスト
	 */
	public static List<String> csvLineToList(String csv, String sepalatorRegex) {
		if (log.isDebugEnabled()) log.debug("-------- start csvLineToList(String csv)");
		
		if (StringUtils.isEmpty(csv)) {
			@SuppressWarnings("unchecked")
			List<String> ret = Collections.EMPTY_LIST;
			return ret;
		}
		
		String[] values = csv.split(sepalatorRegex);
		List<String> list = new ArrayList<String>();
		CollectionUtils.addAll(list, values);
		
		return list;
	}
	
	/**
	 * URLが正しいかどうかを判定します。<br>
	 * （フルパス指定のみ有効）
	 * @param url チェック対象
	 * @return URLとして正しければ true
	 */
	public static boolean isValidUrl(String url) {
		if (log.isDebugEnabled()) log.debug("-------- start isValidUrl(String url)");
		
		if (log.isDebugEnabled()) log.debug("チェック対象文字列[" + url + "]");
		
		if (url != null && url.matches(urlPattern)) {
			return true;
		}
		return false;
	}
	
}
