package jp.co.sample.util;

import java.math.BigDecimal;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import jp.co.sample.contants.StringConstants;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * 時間関連の操作を行うUtilクラス。
 * @author maki oiji
 *
 */
public class DateUtil {
	
	/** logger */
	private static final Log log = LogFactory.getLog(DateUtil.class);

	/** システム日付 */
	private static final String SYS_DATE_KEY = "sys_date";

	/** 日付フォーマット(yyyy/MM/dd HH:mm:ss.sss) */
	private static final DateFormat YYYYMMDDHHMMSSSSS_SLASH = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS");
	/** 日付フォーマット(yyyy-MM-dd HH:mm:ss.sss) */
	private static final DateFormat YYYYMMDDHHMMSSSSS_HYPHEN = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
	/** 日付フォーマット(yyyyMMddHHmmsssss) */
	private static final DateFormat YYYYMMDDHHMMSSSSS = new SimpleDateFormat("yyyyMMddHHmmssSSS");
	/** 日付フォーマット(yyyy/MM/dd HH:mm:ss) */
	private static final DateFormat YYYYMMDDHHMMSS_SLASH = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
	/** 日付フォーマット(yyyy-MM-dd HH:mm:ss) */
	private static final DateFormat YYYYMMDDHHMMSS_HYPHEN = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	/** 日付フォーマット(yyyyMMddHHmmss) */
	private static final DateFormat YYYYMMDDHHMMSS = new SimpleDateFormat("yyyyMMddHHmmss");
	/** 日付フォーマット(yyyy/MM/dd HH:mm) */
	private static final DateFormat YYYYMMDDHHMM_SLASH = new SimpleDateFormat("yyyy/MM/dd HH:mm");
	/** 日付フォーマット(yyyy-MM-dd HH:mm) */
	private static final DateFormat YYYYMMDDHHMM_HYPHEN = new SimpleDateFormat("yyyy-MM-dd HH:mm");
	/** 日付フォーマット(yyyyMMddHHmm) */
	private static final DateFormat YYYYMMDDHHMM = new SimpleDateFormat("yyyyMMddHHmm");
	/** 日付フォーマット(yyyy/MM/dd) */
	private static final DateFormat YYYYMMDD_SLASH = new SimpleDateFormat("yyyy/MM/dd");
	/** 日付フォーマット(yyyy-MM-dd) */
	private static final DateFormat YYYYMMDD_HYPHEN = new SimpleDateFormat("yyyy-MM-dd");
	/** 日付フォーマット(yyyyMMdd) */
	private static final DateFormat YYYYMMDD = new SimpleDateFormat("yyyyMMdd");
	/** 日付フォーマット(yyyy/MM) */
	private static final DateFormat YYYYMM_SLASH = new SimpleDateFormat("yyyy/MM");
	/** 日付フォーマット(yyyy-MM) */
	private static final DateFormat YYYYMM_HYPHEN = new SimpleDateFormat("yyyy-MM");
	/** 日付フォーマット(yyyyMM) */
	private static final DateFormat YYYYMM = new SimpleDateFormat("yyyyMM");
	/** 日付フォーマット(MM/dd) */
	private static final DateFormat MMDD_SLASH = new SimpleDateFormat("MM/dd");
	/** 日付フォーマット(MM-dd) */
	private static final DateFormat MMDD_HYPHEN = new SimpleDateFormat("MM-dd");
	/** 日付フォーマット(MMdd) */
	private static final DateFormat MMDD = new SimpleDateFormat("MMdd");
	/** 日付フォーマット(HH:mm:sssss) */
	private static final DateFormat HHMMSSSSS_COLON = new SimpleDateFormat("HH:mm:ss.SSS");
	/** 日付フォーマット(HHmmsssss) */
	private static final DateFormat HHMMSSSSS = new SimpleDateFormat("HHmmssSSS");
	/** 日付フォーマット(HH:mm:ss) */
	private static final DateFormat HHMMSS_COLON = new SimpleDateFormat("HH:mm:ss");
	/** 日付フォーマット(HHmmss) */
	private static final DateFormat HHMMSS = new SimpleDateFormat("HHmmss");
	/** 日付フォーマット(HH:mm) */
	private static final DateFormat HHMM_COLON = new SimpleDateFormat("HH:mm");
	/** 日付フォーマット(HHmm) */
	private static final DateFormat HHMM = new SimpleDateFormat("HHmm");
	/** 日付フォーマット(yyyy) */
	private static final DateFormat YYYY = new SimpleDateFormat("yyyy");
	/** 日付フォーマット(MM) */
	private static final DateFormat MM = new SimpleDateFormat("MM");
	/** 日付フォーマット(dd) */
	private static final DateFormat DD = new SimpleDateFormat("dd");
	/** 日付フォーマット(HH) */
	private static final DateFormat HH = new SimpleDateFormat("HH");
	/** 日付フォーマット(mm) */
	private static final DateFormat MIN = new SimpleDateFormat("mm");
	/** 日付フォーマット(ss) */
	private static final DateFormat SS = new SimpleDateFormat("ss");

	/**
	 * システム日時を取得する。
	 * @return システム日時
	 */
	public static Date getDate() {
		if (log.isDebugEnabled()) log.debug("-------- start getDate()");
		
		if (System.getProperty(SYS_DATE_KEY) == null) {
			return new Date();
		}
		// テスト時の設定日時
		Date sysDate = parseDateFromYyyyMmDdHhMmSs(System.getProperty(SYS_DATE_KEY));
		return sysDate;
		
	}

	//-------------------------------------------------------------
	// parse
	//-------------------------------------------------------------
	
	/**
	 * "yyyy/MM/dd HH:mm:ss.sss"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy/MM/dd HH:mm:ss.sss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHhMmSsSssSlash(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHhMmSsSssSlash(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSSSSS_SLASH) {
			YYYYMMDDHHMMSSSSS_SLASH.setLenient(false);
			try {
				return YYYYMMDDHHMMSSSSS_SLASH.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy-MM-dd HH:mm:ss.sss"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy-MM-dd HH:mm:ss.sss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHhMmSsSssHyphen(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHhMmSsSssHyphen(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSSSSS_HYPHEN) {
			YYYYMMDDHHMMSSSSS_HYPHEN.setLenient(false);
			try {
				return YYYYMMDDHHMMSSSSS_HYPHEN.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyyMMddHHmmsssss"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyyMMddHHmmsssss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHhMmSsSss(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHhMmSsSss(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSSSSS) {
			YYYYMMDDHHMMSSSSS.setLenient(false);
			try {
				return YYYYMMDDHHMMSSSSS.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy/MM/dd HH:mm:ss"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy/MM/dd HH:mm:ss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHhMmSsSlash(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHhMmSsSlash(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSS_SLASH) {
			YYYYMMDDHHMMSS_SLASH.setLenient(false);
			try {
				return YYYYMMDDHHMMSS_SLASH.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy-MM-dd HH:mm:ss"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy-MM-dd HH:mm:ss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHhMmSsHyphen(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHhMmSsHyphen(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSS_HYPHEN) {
			YYYYMMDDHHMMSS_HYPHEN.setLenient(false);
			try {
				return YYYYMMDDHHMMSS_HYPHEN.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyyMMddHHmmss"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyyMMddHHmmss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHhMmSs(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHhMmSs(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSS) {
			YYYYMMDDHHMMSS.setLenient(false);
			try {
				return YYYYMMDDHHMMSS.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy/MM/dd HH:mm"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy/MM/dd HH:mm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHhMmSlash(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHhMmSlash(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDDHHMM_SLASH) {
			YYYYMMDDHHMM_SLASH.setLenient(false);
			try {
				return YYYYMMDDHHMM_SLASH.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy-MM-dd HH:mm"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy-MM-dd HH:mm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHhMmHyphen(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHhMmHyphen(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDDHHMM_HYPHEN) {
			YYYYMMDDHHMM_HYPHEN.setLenient(false);
			try {
				return YYYYMMDDHHMM_HYPHEN.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyyMMddHHmm"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyyMMddHHmm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHhMm(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHhMm(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDDHHMM) {
			YYYYMMDDHHMM.setLenient(false);
			try {
				return YYYYMMDDHHMM.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy/MM/dd"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy/MM/dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdSlash(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdSlash(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDD_SLASH) {
			YYYYMMDD_SLASH.setLenient(false);
			try {
				return YYYYMMDD_SLASH.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy-MM-dd"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy-MM-dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDdHyphen(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDdHyphen(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDD_HYPHEN) {
			YYYYMMDD_HYPHEN.setLenient(false);
			try {
				return YYYYMMDD_HYPHEN.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyyMMdd"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyyMMdd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmDd(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmDd(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMMDD) {
			YYYYMMDD.setLenient(false);
			try {
				return YYYYMMDD.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy/MM"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy/MM"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmSlash(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmSlash(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMM_SLASH) {
			YYYYMM_SLASH.setLenient(false);
			try {
				return YYYYMM_SLASH.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy-MM"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy-MM"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMmHyphen(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMmHyphen(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMM_HYPHEN) {
			YYYYMM_HYPHEN.setLenient(false);
			try {
				return YYYYMM_HYPHEN.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyyMM"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyyMM"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyyMm(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyyMm(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYYMM) {
			YYYYMM.setLenient(false);
			try {
				return YYYYMM.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "MM/dd"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "MM/dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromMmDdSlash(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromMmDdSlash(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (MMDD_SLASH) {
			MMDD_SLASH.setLenient(false);
			try {
				return MMDD_SLASH.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "MM-dd"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "MM-dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromMmDdHyphen(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromMmDdHyphen(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (MMDD_HYPHEN) {
			MMDD_HYPHEN.setLenient(false);
			try {
				return MMDD_HYPHEN.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "MMdd"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "MMdd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromMmDd(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromMmDd(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (MMDD) {
			MMDD.setLenient(false);
			try {
				return MMDD.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "HH:mm:sssss"形式の文字列から時間をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "HH:mm:sssss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseTimeFromHhMmSsSssColon(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTimeFromHhMmSsSssColon(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (HHMMSSSSS_COLON) {
			HHMMSSSSS_COLON.setLenient(false);
			try {
				return HHMMSSSSS_COLON.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "HHmmsssss"形式の文字列から時間をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "HHmmsssss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseTimeFromHhMmSsSss(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTimeFromHhMmSsSss(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (HHMMSSSSS) {
			HHMMSSSSS.setLenient(false);
			try {
				return HHMMSSSSS.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "HH:mm:ss"形式の文字列から時間をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "HH:mm:ss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseTimeFromHhMmSsColon(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTimeFromHhMmSsColon(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (HHMMSS_COLON) {
			HHMMSS_COLON.setLenient(false);
			try {
				return HHMMSS_COLON.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "HHmmss"形式の文字列から時間をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "HHmmss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseTimeFromHhMmSs(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTimeFromHhMmSs(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (HHMMSS) {
			HHMMSS.setLenient(false);
			try {
				return HHMMSS.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "HH:mm"形式の文字列から時間をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "HH:mm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseTimeFromHhMmColon(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTimeFromHhMmColon(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (HHMM_COLON) {
			HHMM_COLON.setLenient(false);
			try {
				return HHMM_COLON.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "HHmm"形式の文字列から時間をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "HHmm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseTimeFromHhMm(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTimeFromHhMm(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (HHMM) {
			HHMM.setLenient(false);
			try {
				return HHMM.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "yyyy"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "yyyy"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromYyyy(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromYyyy(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (YYYY) {
			YYYY.setLenient(false);
			try {
				return YYYY.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "MM"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "MM"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromMm(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromMm(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (MM) {
			MM.setLenient(false);
			try {
				return MM.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "dd"形式の文字列から日付をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseDateFromDd(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateFromDd(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (DD) {
			DD.setLenient(false);
			try {
				return DD.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "HH"形式の文字列から時間をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "HH"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseTimeFromHh(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTimeFromHh(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (HH) {
			HH.setLenient(false);
			try {
				return HH.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "mm"形式の文字列から時間をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "mm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseTimeFromMm(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTimeFromMm(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (MIN) {
			MIN.setLenient(false);
			try {
				return MIN.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * "ss"形式の文字列から時間をDate型で取得します。<BR>
	 * 変換に失敗した場合は、nullを返します。
	 * @param dateString "ss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static Date parseTimeFromSs(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTimeFromSs(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		synchronized (SS) {
			SS.setLenient(false);
			try {
				return SS.parse(dateString);
			} catch (ParseException e) {
				if (log.isDebugEnabled()) log.debug("parse処理でExceptionが発生しました。対象文字列：" + dateString, e);
				return null;
			}
		}
	}

	/**
	 * 年、月、日、時、分、秒の文字列からDate型を取得します。
	 * @param year 年文字列
	 * @param month 月文字列
	 * @param day 日文字列
	 * @param hour 時文字列
	 * @param min 分文字列
	 * @param sec 秒文字列
	 * @return 生成されたDate型
	 */
	public static Date parseDate(String year, String month, String day, String hour, String min, String sec) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDate(String year, String month, String day, String hour, String min, String sec)");

		StringBuilder sb = new StringBuilder();
		sb.append(year);
		sb.append(StringConstants.SLASH);
		sb.append(month);
		sb.append(StringConstants.SLASH);
		sb.append(day);
		sb.append(StringConstants.SPACE);
		sb.append(hour);
		sb.append(StringConstants.COLON);
		sb.append(min);
		sb.append(StringConstants.COLON);
		sb.append(sec);
		return parseDateFromYyyyMmDdHhMmSsSlash(sb.toString());
	}

	/**
	 * 年、月、日の文字列からDate型を取得します。
	 * @param year 年文字列
	 * @param month 月文字列
	 * @param day 日文字列
	 * @return 生成されたDate型
	 */
	public static Date parseDate(String year, String month, String day) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDate(String year, String month, String day)");

		StringBuilder sb = new StringBuilder();
		sb.append(year);
		sb.append(StringConstants.SLASH);
		sb.append(month);
		sb.append(StringConstants.SLASH);
		sb.append(day);
		return parseDateFromYyyyMmDdSlash(sb.toString());
	}

	/**
	 * 年、月の文字列からDate型を取得します。
	 * @param year 年文字列
	 * @param month 月文字列
	 * @return 生成されたDate型
	 */
	public static Date parseDateYearMonth(String year, String month) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateYearMonth(String year, String month)");

		StringBuilder sb = new StringBuilder();
		sb.append(year);
		sb.append(StringConstants.SLASH);
		sb.append(month);
		return parseDateFromYyyyMmSlash(sb.toString());
	}

	/**
	 * 月、日の文字列からDate型を取得します。
	 * @param month 月文字列
	 * @param day 日文字列
	 * @return 生成されたDate型
	 */
	public static Date parseDateMonthDay(String month, String day) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDateMonthDay(String month, String day)");

		StringBuilder sb = new StringBuilder();
		sb.append(month);
		sb.append(StringConstants.SLASH);
		sb.append(day);
		return parseDateFromMmDdSlash(sb.toString());
	}

	/**
	 * 時、分、秒の文字列からDate型を取得します。
	 * @param hour 時文字列
	 * @param min 分文字列
	 * @param sec 秒文字列
	 * @return 生成されたDate型
	 */
	public static Date parseTime(String hour, String min, String sec) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTime(String hour, String min, String sec)");

		StringBuilder sb = new StringBuilder();
		sb.append(hour);
		sb.append(StringConstants.COLON);
		sb.append(min);
		sb.append(StringConstants.COLON);
		sb.append(sec);
		return parseTimeFromHhMmSsColon(sb.toString());
	}

	/**
	 * 時、分の文字列からDate型を取得します。
	 * @param hour 時文字列
	 * @param min 分文字列
	 * @return 生成されたDate型
	 */
	public static Date parseTime(String hour, String min) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTime(String hour, String min)");

		StringBuilder sb = new StringBuilder();
		sb.append(hour);
		sb.append(StringConstants.COLON);
		sb.append(min);
		return parseTimeFromHhMmColon(sb.toString());
	}

	/**
	 * 日付文字列をDate型に変換します。<BR>
	 * 対応形式は"yyyy/MM/dd"、"yyyy-MM-dd"、"yyyyMMdd"の3種類。
	 * @param dateString 日付文字列
	 * @return 生成されたDate型
	 */
	public static Date parseDate(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseDate(String dateString)");

		if (StringUtils.isEmpty(dateString)) {
			return null;
		}
		if (dateString.length() == 8) {
			return parseDateFromYyyyMmDd(dateString);
		} else if (dateString.length() == 10) {
			String yyyyMMdd = dateString.replace(StringConstants.HYPHEN, StringConstants.SLASH);
			return parseDateFromYyyyMmDdSlash(yyyyMMdd);
		}
		return null;
	}

	/**
	 * 時間文字列をDate型に変換します。<BR>
	 * 対応形式は"HH:mm:ss.SSS"、"HHmmssSSS"、"HH:mm:ss"、"HHmmss"、"HH:mm"、"HHmm"。
	 * @param timeString 時間文字列
	 * @return 生成されたDate型
	 */
	public static Date parseTime(String timeString) {
		if (log.isDebugEnabled()) log.debug("-------- start parseTime(String timeString)");

		if (StringUtils.isEmpty(timeString)) {
			return null;
		}
		if (timeString.length() == 12) {
			return parseTimeFromHhMmSsSssColon(timeString);
		} else if (timeString.length() == 9) {
			return parseTimeFromHhMmSsSss(timeString);
		} else if (timeString.length() == 8) {
			return parseTimeFromHhMmSsColon(timeString);
		} else if (timeString.length() == 6) {
			return parseTimeFromHhMmSs(timeString);
		} else if (timeString.length() == 5) {
			return parseTimeFromHhMmColon(timeString);
		} else if (timeString.length() == 4) {
			return parseTimeFromHhMm(timeString);
		}
		return null;
	}


	//-------------------------------------------------------------
	// format
	//-------------------------------------------------------------
	
	/**
	 * Date型からyyyy/MM/dd HH:mm:ss.sss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy/MM/dd HH:mm:ss.sss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHhMmSsSssSlash(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHhMmSsSssSlash(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSSSSS_SLASH) {
			return YYYYMMDDHHMMSSSSS_SLASH.format(date);
		}
	}

	/**
	 * Date型からyyyy-MM-dd HH:mm:ss.sss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy-MM-dd HH:mm:ss.sss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHhMmSsSssHyphen(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHhMmSsSssHyphen(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSSSSS_HYPHEN) {
			return YYYYMMDDHHMMSSSSS_HYPHEN.format(date);
		}
	}

	/**
	 * Date型からyyyyMMddHHmmsssss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyyMMddHHmmsssss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHhMmSsSss(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHhMmSsSss(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSSSSS) {
			return YYYYMMDDHHMMSSSSS.format(date);
		}
	}

	/**
	 * Date型からyyyy/MM/dd HH:mm:ss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy/MM/dd HH:mm:ss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHhMmSsSlash(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHhMmSsSlash(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSS_SLASH) {
			return YYYYMMDDHHMMSS_SLASH.format(date);
		}
	}

	/**
	 * Date型からyyyy-MM-dd HH:mm:ss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy-MM-dd HH:mm:ss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHhMmSsHyphen(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHhMmSsHyphen(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSS_HYPHEN) {
			return YYYYMMDDHHMMSS_HYPHEN.format(date);
		}
	}

	/**
	 * Date型からyyyyMMddHHmmss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyyMMddHHmmss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHhMmSs(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHhMmSs(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDDHHMMSS) {
			return YYYYMMDDHHMMSS.format(date);
		}
	}

	/**
	 * Date型からyyyy/MM/dd HH:mm"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy/MM/dd HH:mm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHhMmSlash(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHhMmSlash(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDDHHMM_SLASH) {
			return YYYYMMDDHHMM_SLASH.format(date);
		}
	}

	/**
	 * Date型からyyyy-MM-dd HH:mm"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy-MM-dd HH:mm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHhMmHyphen(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHhMmHyphen(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDDHHMM_HYPHEN) {
			return YYYYMMDDHHMM_HYPHEN.format(date);
		}
	}

	/**
	 * Date型からyyyyMMddHHmm"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyyMMddHHmm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHhMm(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHhMm(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDDHHMM) {
			return YYYYMMDDHHMM.format(date);
		}
	}

	/**
	 * Date型からyyyy/MM/dd"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy/MM/dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdSlash(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdSlash(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDD_SLASH) {
			return YYYYMMDD_SLASH.format(date);
		}
	}

	/**
	 * Date型からyyyy-MM-dd"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy-MM-dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDdHyphen(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDdHyphen(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDD_HYPHEN) {
			return YYYYMMDD_HYPHEN.format(date);
		}
	}

	/**
	 * Date型からyyyyMMdd"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyyMMdd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmDd(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmDd(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMMDD) {
			return YYYYMMDD.format(date);
		}
	}

	/**
	 * Date型からyyyy/MM"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy/MM"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmSlash(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmSlash(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMM_SLASH) {
			return YYYYMM_SLASH.format(date);
		}
	}

	/**
	 * Date型からyyyy-MM"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy-MM"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMmHyphen(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMmHyphen(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMM_HYPHEN) {
			return YYYYMM_HYPHEN.format(date);
		}
	}

	/**
	 * Date型からyyyyMM"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyyMM"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyyMm(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyyMm(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYYMM) {
			return YYYYMM.format(date);
		}
	}

	/**
	 * Date型からMM/dd"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "MM/dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToMmDdSlash(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToMmDdSlash(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (MMDD_SLASH) {
			return MMDD_SLASH.format(date);
		}
	}

	/**
	 * Date型からMM-dd"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "MM-dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToMmDdHyphen(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToMmDdHyphen(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (MMDD_HYPHEN) {
			return MMDD_HYPHEN.format(date);
		}
	}

	/**
	 * Date型からMMdd"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "MMdd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToMmDd(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToMmDd(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (MMDD) {
			return MMDD.format(date);
		}
	}

	/**
	 * Date型からHH:mm:sssss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "HH:mm:sssss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatTimeToHhMmSsSssColon(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatTimeToHhMmSsSssColon(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (HHMMSSSSS_COLON) {
			return HHMMSSSSS_COLON.format(date);
		}
	}

	/**
	 * Date型からHHmmsssss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "HHmmsssss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatTimeToHhMmSsSss(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatTimeToHhMmSsSss(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (HHMMSSSSS) {
			return HHMMSSSSS.format(date);
		}
	}

	/**
	 * Date型からHH:mm:ss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "HH:mm:ss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatTimeToHhMmSsColon(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatTimeToHhMmSsColon(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (HHMMSS_COLON) {
			return HHMMSS_COLON.format(date);
		}
	}

	/**
	 * Date型からHHmmss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "HHmmss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatTimeToHhMmSs(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatTimeToHhMmSs(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (HHMMSS) {
			return HHMMSS.format(date);
		}
	}

	/**
	 * Date型からHH:mm"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "HH:mm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatTimeToHhMmColon(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatTimeToHhMmColon(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (HHMM_COLON) {
			return HHMM_COLON.format(date);
		}
	}

	/**
	 * Date型からHHmm"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "HHmm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatTimeToHhMm(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatTimeToHhMm(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (HHMM) {
			return HHMM.format(date);
		}
	}

	/**
	 * Date型からyyyy"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "yyyy"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToYyyy(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToYyyy(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (YYYY) {
			return YYYY.format(date);
		}
	}

	/**
	 * Date型からMM"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "MM"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToMm(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToMm(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (MM) {
			return MM.format(date);
		}
	}

	/**
	 * Date型からdd"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "dd"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatDateToDd(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatDateToDd(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (DD) {
			return DD.format(date);
		}
	}

	/**
	 * Date型からHH"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "HH"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatTimeToHh(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatTimeToHh(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (HH) {
			return HH.format(date);
		}
	}

	/**
	 * Date型からmm"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "mm"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatTimeToMm(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatTimeToMm(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (MIN) {
			return MIN.format(date);
		}
	}

	/**
	 * Date型からss"形式の文字列を取得します。
	 * <BR>指定日付がnullの場合nullを返します。
	 * @param date "ss"形式の日付文字列
	 * @return 指定された文字列を変換したDate型
	 */
	public static String formatTimeToSs(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start formatTimeToSs(Date date)");

		if (date == null) {
			return null;
		}
		synchronized (SS) {
			return SS.format(date);
		}
	}
	
	
	//-------------------------------------------------------------
	// 日付計算関連
	//-------------------------------------------------------------

	/**
	 * 対象の日付に指定された年数を加算します。
	 * @param date 加算対象日付
	 * @param amount 加算数
	 * @return 計算された日付
	 */
	public static Date addYear(Date date, int amount) {
		if (log.isDebugEnabled()) log.debug("-------- start addYear(Date date, int amount)");

		return calcDate(date, Calendar.YEAR, amount);
	}
	
	/**
	 * 対象の日付に指定された月数を加算します。
	 * @param date 加算対象日付
	 * @param amount 加算数
	 * @return 計算された日付
	 */
	public static Date addMonth(Date date, int amount) {
		if (log.isDebugEnabled()) log.debug("-------- start addMonth(Date date, int amount)");

		return calcDate(date, Calendar.MONTH, amount);
	}

	/**
	 * 対象の日付に指定された日数を加算します。
	 * @param date 加算対象日付
	 * @param amount 加算数
	 * @return 計算された日付
	 */
	public static Date addDate(Date date, int amount) {
		if (log.isDebugEnabled()) log.debug("-------- start addDate(Date date, int amount)");

		return calcDate(date, Calendar.DATE, amount);
	}

	/**
	 * 対象の日時に指定された時間数を加算します。
	 * @param date 加算対象日時
	 * @param amount 加算数
	 * @return 計算された日時
	 */
	public static Date addHour(Date date, int amount) {
		if (log.isDebugEnabled()) log.debug("-------- start addHour(Date date, int amount)");

		return calcDate(date, Calendar.HOUR_OF_DAY, amount);
	}

	/**
	 * 対象の日時に指定された分数を加算します。
	 * @param date 加算対象日時
	 * @param amount 加算数
	 * @return 計算された日時
	 */
	public static Date addMinute(Date date, int amount) {
		if (log.isDebugEnabled()) log.debug("-------- start addMinute(Date date, int amount)");

		return calcDate(date, Calendar.MINUTE, amount);
	}

	/**
	 * 対象の日時に指定された秒数を加算します。
	 * @param date 加算対象日時
	 * @param amount 加算数
	 * @return 計算された日時
	 */
	public static Date addSecond(Date date, int amount) {
		if (log.isDebugEnabled()) log.debug("-------- start addSecond(Date date, int amount)");

		return calcDate(date, Calendar.SECOND, amount);
	}

	/**
	 * 対象の日付に指定されたフィールドを指定数加算します。
	 * @param date 加算対象日付
	 * @param field 加算するフィールドを表す数値
	 * @param amount 加算数
	 * @return 計算された日付
	 */
	private static Date calcDate(Date date, int field, int amount) {
		
		if (date == null) {
			return null;
		}
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.add(field, amount);
		return cal.getTime();
	}
	
	/**
	 * 曜日を取得します。
	 * <BR>日付がnullの場合はシステム日付の曜日を返します。
	 * @param date 日付
	 * @return Calendar型のstaticフィールドの曜日定数
	 */
	public static int getDayOfWeek(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start getDayOfWeek(Date date)");

		if (date == null) {
			date = DateUtil.getDate();
		}
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		return cal.get(Calendar.DAY_OF_WEEK);
	}

	/**
	 * 指定された日付の月の月初日をDate型で取得します。<BR>
	 * 日付がnullの場合は当月の初日を取得します。
	 * @param date 日付
	 * @return 指定月の月初日
	 */
	public static Date getMonthFirstDate(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start getMonthFirstDate(Date date)");

		if (date == null) {
			date = DateUtil.getDate();
		}
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.set(Calendar.DATE, cal.getActualMinimum(Calendar.DATE));
		return cal.getTime();
	}
	
	/**
	 * 指定された日付の月の前月の1日のDate型を取得します。<BR>
	 * 日付がnullの場合は当月の前月の1日を取得します。
	 * @param date 日付
	 * @return 指定月の前月の月初日
	 */
	public static Date getLastMonthFirstDate(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start getLastMonthFirstDate(Date date)");

		return addMonth(getMonthFirstDate(date), -1);
	}

	/**
	 * 指定された日付の月の翌月の1日のDate型を取得します。<BR>
	 * 日付がnullの場合は当月の翌月の1日を取得します。
	 * @param date 日付
	 * @return 指定月の翌月の月初日
	 */
	public static Date getNextMonthFirstDate(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start getNextMonthFirstDate(Date date)");

		return addMonth(getMonthFirstDate(date), 1);
	}

	/**
	 * 指定された日付の月の月末日をDate型で取得します。<BR>
	 * 日付がnullの場合は当月の末日を取得します。
	 * @param date 日付
	 * @return 指定月の月末日
	 */
	public static Date getMonthLastDate(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start getMonthLastDate(Date date)");

		if (date == null) {
			date = DateUtil.getDate();
		}
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.set(Calendar.DATE, cal.getActualMaximum(Calendar.DATE));
		return cal.getTime();
	}

	/**
	 * 指定された日付の前月の月末日をDate型で取得します。<BR>
	 * 日付がnullの場合は当月の前月の末日を取得します。
	 * @param date 日付
	 * @return 指定月の前月の月末日
	 */
	public static Date getLastMonthLastDate(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start getLastMonthLastDate(Date date)");

		return addDate(getMonthFirstDate(date), -1);
	}

	/**
	 * 指定された日付の翌月の月末日をDate型で取得します。<BR>
	 * 日付がnullの場合は当月の翌月の末日を取得します。
	 * @param date 日付
	 * @return 指定月の翌月の月末日
	 */
	public static Date getNextMonthLastDate(Date date) {
		if (log.isDebugEnabled()) log.debug("-------- start getNextMonthLastDate(Date date)");

		Calendar cal = Calendar.getInstance();
		cal.setTime(getNextMonthFirstDate(date));
		cal.set(Calendar.DATE, cal.getActualMaximum(Calendar.DATE));
		return cal.getTime();
	}


	/**
	 * 誕生日から現在の年齢を計算します。<BR>
	 * 誕生日がnullの場合、0を返します。
	 * @param birthday 誕生日
	 * @return 年齢
	 */
	public static int calcAge(Date birthday) {
		if (log.isDebugEnabled()) log.debug("-------- start calcAge(Date birthday)");

		if (birthday == null) {
			return 0;
		}
		return calcYearInterval(getDate(), birthday);
	}
	
	/**
	 * 二つの日付の間の年数を取得します。（小数点以下切り捨て）<BR>
	 * なお、二つの日付の両方がnullの場合は0を返します。<BR>
	 * どちらかがnullの場合はシステム日時との差分を取得します。<BR>
	 * 二つの日付の間にある差分を計算するため、負の数になることはありません。
	 * @param date0 対象日付
	 * @param date1 対象日付
	 * @return 二つの日付の差分（年）
	 */
	public static int calcYearInterval(Date date0, Date date1) {
		if (log.isDebugEnabled()) log.debug("-------- start calcYearInterval(Date date0, Date date1)");

		if (date0 == null && date1 == null) {
			return 0;
		} else if (date0 == null) {
			date0 = getDate();
		} else if (date1 == null) {
			date1 = getDate();
		}
		int interval = (Integer.parseInt(formatDateToYyyyMmDd(date0)) - Integer.parseInt(formatDateToYyyyMmDd(date1))) / 10000;
		if (interval < 0) {
			return interval * -1;
		}
		return interval;
	}

	/**
	 * 二つの日付の間の日数を取得します。（小数点以下切り捨て）<BR>
	 * なお、二つの日付の両方がnullの場合は0を返します。<BR>
	 * どちらかがnullの場合はシステム日時との差分を取得します。<BR>
	 * 二つの日付の間にある差分を計算するため、負の数になることはありません。
	 * @param date0 対象日付
	 * @param date1 対象日付
	 * @return 二つの日付の差分（日）
	 */
	public static int calcDayInterval(Date date0, Date date1) {
		if (log.isDebugEnabled()) log.debug("-------- start calcDayInterval(Date date0, Date date1)");

		return new BigDecimal(calcHourInterval(date0, date1)).divide(new BigDecimal(24)).intValue();
	}

	/**
	 * 二つの日付の間の時間数を取得します。（小数点以下切り捨て）<BR>
	 * なお、二つの日付の両方がnullの場合は0を返します。<BR>
	 * どちらかがnullの場合はシステム日時との差分を取得します。<BR>
	 * 二つの日付の間にある差分を計算するため、負の数になることはありません。
	 * @param date0 対象日付
	 * @param date1 対象日付
	 * @return 二つの日付の差分（時）
	 */
	public static int calcHourInterval(Date date0, Date date1) {
		if (log.isDebugEnabled()) log.debug("-------- start calcHourInterval(Date date0, Date date1)");

		return new BigDecimal(calcMinuteInterval(date0, date1)).divide(new BigDecimal(60)).intValue();
	}

	/**
	 * 二つの日付の間の分数を取得します。（小数点以下切り捨て）<BR>
	 * なお、二つの日付の両方がnullの場合は0を返します。<BR>
	 * どちらかがnullの場合はシステム日時との差分を取得します。<BR>
	 * 二つの日付の間にある差分を計算するため、負の数になることはありません。
	 * @param date0 対象日付
	 * @param date1 対象日付
	 * @return 二つの日付の差分（分）
	 */
	public static long calcMinuteInterval(Date date0, Date date1) {
		if (log.isDebugEnabled()) log.debug("-------- start calcMinuteInterval(Date date0, Date date1)");

		return new BigDecimal(calcSecondInterval(date0, date1)).divide(new BigDecimal(60)).longValue();
	}

	/**
	 * 二つの日付の間の秒数を取得します。（小数点以下切り捨て）<BR>
	 * なお、二つの日付の両方がnullの場合は0を返します。<BR>
	 * どちらかがnullの場合はシステム日時との差分を取得します。<BR>
	 * 二つの日付の間にある差分を計算するため、負の数になることはありません。
	 * @param date0 対象日付
	 * @param date1 対象日付
	 * @return 二つの日付の差分（秒）
	 */
	public static long calcSecondInterval(Date date0, Date date1) {
		if (log.isDebugEnabled()) log.debug("-------- start calcSecondInterval(Date date0, Date date1)");

		if (date0 == null && date1 == null) {
			return 0;
		} else if (date0 == null) {
			date0 = getDate();
		} else if (date1 == null) {
			date1 = getDate();
		}
		BigDecimal num0 = new BigDecimal(date0.getTime());
		BigDecimal num1 = new BigDecimal(date1.getTime());
		long interval = num0.add(num1.multiply(new BigDecimal(-1))).divide(new BigDecimal(1000)).longValue();
		if (interval < 0) {
			return interval * -1;
		}
		return interval;
	}
	
	
	//-------------------------------------------------------------
	// チェック関連
	//-------------------------------------------------------------
	
	/**
	 * 指定日が範囲内に含まれているかチェックします。<BR>
	 * もし、範囲の開始日が無ければ終了日のみでチェックし、
	 * 範囲の終了日が無ければ開始日のみでチェックします。<BR>
	 * 精度はミリ秒まで考慮します。<BR>
	 * なお、閾値と一致する場合は含まれるものと判定します。
	 * @param target 検査対象日付
	 * @param from 期間開始
	 * @param to 期間終了
	 * @return 範囲内に指定日が含まれていたらtrue
	 */
	public static boolean isInRangeDate(Date target, Date from, Date to) {
		if (log.isDebugEnabled()) log.debug("-------- start isInRangeDate(Date target, Date from, Date to)");

		// 対象日付がnullならfalse
		if (target == null) {
			return false;
		}
		// 範囲の指定が無ければtrue
		if (from == null && to == null) {
			return true;
		}
		// 範囲の指定が逆になっていたら正しく入れ替えて処理
		if (from != null && to != null && from.after(to)) {
			return isInRangeDate(target, to, from);
		}
		// 範囲の開始が指定されており、かつ対象日付が開始日より前ならfalse
		if (from != null && target.before(from)) {
			return false;
		} 
		// 範囲の終了が指定されており、かつ対象日付が終了日より後ならfalse
		if (to != null && target.after(to)) {
			return false;
		}
		return true;
	}
	
	/**
	 * 二つの期間を比較し、重複する期間があるかチェックします。<BR>
	 * それぞれの期間の開始・終了のいずれかがnullであればfalseを返します。<BR>
	 * 精度はミリ秒まで考慮します。
	 * @param term1From 期間１開始
	 * @param term1To 期間１終了
	 * @param term2From 期間２開始
	 * @param term2To 期間２終了
	 * @return 重複する期間が少しでもあればtrue
	 */
	public static boolean isOverlapTerm(Date term1From, Date term1To, Date term2From, Date term2To) {
		if (log.isDebugEnabled()) log.debug("-------- start isOverlapTerm(Date term1From, Date term1To, Date term2From, Date term2To)");
		
		if (term1From == null || term1To == null || term2From == null || term2To == null) {
			return false;
		}
		// 期間１開始が期間１終了以降なら入れ替える
		if (term1From.compareTo(term1To) > 0) {
			return isOverlapTerm(term1To, term1From, term2From, term2To);
		}
		// 期間２開始が期間２終了以降なら入れ替える
		if (term2From.compareTo(term2To) > 0) {
			return isOverlapTerm(term1From, term1To, term2To, term2From);
		}
		// 順序が正しいはずなので期間１開始が期間２終了より後なら重ならない
		if (term1From.after(term2To)) {
			return false;
		}
		// 順序が正しいはずなので期間１終了が期間２開始より前なら重ならない
		if (term1To.before(term2From)) {
			return false;
		}
		return true;
	}

	/**
	 * 期間１が期間２に含まれるかかチェックします。<BR>
	 * それぞれの期間の開始・終了のいずれかがnullであればfalseを返します。<BR>
	 * 精度はミリ秒まで考慮します。
	 * @param term1From 期間１開始
	 * @param term1To 期間１終了
	 * @param term2From 期間２開始
	 * @param term2To 期間２終了
	 * @return 期間１が期間２に全て含まれていればtrue
	 */
	public static boolean isInRangeTerm(Date term1From, Date term1To, Date term2From, Date term2To) {
		if (log.isDebugEnabled()) log.debug("-------- start isInRangeTerm(Date term1From, Date term1To, Date term2From, Date term2To)");
		
		if (term1From == null || term1To == null || term2From == null || term2To == null) {
			return false;
		}
		if (isInRangeDate(term1From, term2From, term2To) && isInRangeDate(term1To, term2From, term2To)) {
			return true;
		}
		return false;
	}

	/**
	 * 日付妥当性判定。<BR>
	 * 対応形式は"yyyy/MM/dd"、"yyyy-MM-dd"、"yyyyMMdd"の3種類。
	 * @param dateString 日付文字列
	 * @return 日付として正しければtrue
	 */
	public static boolean isValidDate(String dateString) {
		if (log.isDebugEnabled()) log.debug("-------- start isValidDate(String dateString)");
		
		Date date = parseDate(dateString);
		if (date == null) {
			return false;
		}
		return true;
	}

	/**
	 * 日付妥当性判定。<BR>
	 * @param yyyy 年文字列
	 * @param mm 月文字列
	 * @param dd 日文字列
	 * @return 日付として正しければtrue
	 */
	public static boolean isValidDate(String yyyy, String mm, String dd) {
		if (log.isDebugEnabled()) log.debug("-------- start isValidDate(String yyyy, String mm, String dd)");

		return isValidDate(yyyy + StringConstants.SLASH + mm + StringConstants.SLASH + dd);
	}
	
	/**
	 * 時間妥当性判定。<BR>
	 * 対応形式は"HH:mm:ss.SSS"、"HHmmssSSS"、"HH:mm:ss"、"HHmmss"、"HH:mm"、"HHmm"。
	 * @param timeString 時間文字列
	 * @return 時間として正しければtrue
	 */
	public static boolean isValidTime(String timeString) {
		if (log.isDebugEnabled()) log.debug("-------- start isValidTime(String timeString)");

		Date time = parseTime(timeString);
		if (time != null) {
			return true;
		}
		return false;
	}

	//-------------------------------------------------------------
	// 情報取得系
	//-------------------------------------------------------------

	
	/**
	 * 年のリストを取得します。
	 * 引数はそれぞれ、システム日付を起点に何年前（後）から何年前（後）まで、
	 * 取得するかを指定する。<BR>
	 * from <= toの場合は昇順でで取得します。、from > toの場合は降順で取得します。
	 * 例) 50年前～10年後までを取得する場合 getYeayList(-50, 10)
	 * @param from 開始年。システム日付からn年前（後）
	 * @param to 終了年。システム日付からn年前（後）
	 * @return 年のリスト
	 */
	public static List<String> getYeayList(int from, int to) {
		if (log.isDebugEnabled()) log.debug("-------- start getYeayList(int from, int to)");
		
		if (from > to) {
			List<String> list = getYeayList(to, from);
			Collections.reverse(list);
			return list;
		}
		
		List<String> list = new ArrayList<String>();
		
		SimpleDateFormat df = new SimpleDateFormat("yyyy");
		
		Calendar cal = Calendar.getInstance();
		cal.setTime(getDate());
		cal.add(Calendar.YEAR, from);
		
		while (from <= to) {
			list.add(df.format(cal.getTime()));
			cal.add(Calendar.YEAR, 1);
			from++;
		}
		
		return list;
	}

	/**
	 * 指定期間の日付リストを取得します。<BR>
	 * 対応形式は"yyyy/MM/dd"、"yyyy-MM-dd"、"yyyyMMdd"の3種類。<BR>
	 * from <= toの場合は昇順でで取得します。、from > toの場合は降順で取得します。
	 * @param dateFrom 開始日付文字列
	 * @param dateTo 終了日付文字列
	 * @return 日付リスト
	 */
	public static List<Date> getDays(String dateFrom, String dateTo) {
		if (log.isDebugEnabled()) log.debug("-------- start getDays(String dateFrom, String dateTo)");

		if (StringUtils.isEmpty(dateFrom) && StringUtils.isEmpty(dateTo)) {
			return new ArrayList<Date>();
		}
		if (!isValidDate(dateFrom) || !isValidDate(dateTo)) {
			// 不正な日付の場合
			return new ArrayList<Date>();
		}

		// Date型に変換
		// From
		Date from = parseDate(dateFrom);
		// To
		Date to = parseDate(dateTo);

		return getDays(from, to);
	}

	/**
	 * 指定期間の日付リストを取得します。<BR>
	 * from <= toの場合は昇順でで取得します。、from > toの場合は降順で取得します。
	 * @param from 開始日付
	 * @param to 終了日付
	 * @return 日付リスト
	 */
	public static List<Date> getDays(Date from, Date to) {
		if (log.isDebugEnabled()) log.debug("-------- start getDays(Date from, Date to)");

		if (from == null || to == null) {
			return new ArrayList<Date>();
		}
		
		// from・toが逆なら入れ替えて取得
		if (to.compareTo(from) < 0) {
			List<Date> days = getDays(to, from);
			Collections.reverse(days);
			return days;
		}

		List<Date> days = new ArrayList<Date>();

		// カレンダーをFrom日付に設定
		Date d = from;

		while (d.compareTo(to) <= 0) {
			days.add(d);
			d = addDate(d, 1);
		}
		return days;
	}
	
}
